module net.BurtonRadons.parse.type;

class Type
{
    import net.BurtonRadons.parse.expression;
    import net.BurtonRadons.parse.lexer;
    import net.BurtonRadons.parse.scope;
    
    Type next; /**< Next type or null for the root type. */

    static Type findBasicType (char [] name)
    {
        switch (name)
        {
            case "bit": return BitType.singleton;
            case "byte": return ByteType.singleton;
            case "char": return CharType.singleton;
            case "complex": return ComplexType.singleton;
            case "double": return DoubleType.singleton;
            case "real": return RealType.singleton;
            case "float": return FloatType.singleton;
            case "int": return IntType.singleton;
            case "long": return LongType.singleton;
            case "ubyte": return UByteType.singleton;
            case "uint": return UIntType.singleton;
            case "ulong": return ULongType.singleton;
            case "ushort": return UShortType.singleton;
            case "void": return VoidType.singleton;
            default:
                throw new Error (fmt ("Unknown type for findBasicType: %.*s", name));
                return null;
        }
    }

    abstract int cellSize (); /**< The number of bytes in each cell of the instance. */
    bit isReal () { return false; } /**< Whether this is a real type (float, double, real, complex, or imaginary). */
    bit isBaseReal () { return false; } /**< Whether this is float, double, or real. */
    bit isInteger () { return false; } /**< Whether this is an integer type (byte, ubyte, short, ushort, int, uint, long, ulong, cent, ucent). */
    override int opEqual (Object b) { return this.classinfo === b.classinfo; }
    int alignSize () { return cellSize () < 4 ? cellSize () : 4; }
    int alignValue (int value) { int a = alignSize (); return (value + a - 1) / a * a; }
    int alignTo (int size) { return (cellSize () + size - 1) / size * size; }
    override char [] toString () { throw new Error (this.classinfo.name ~ ".toString ()"); return null; }

    /** Return the type that should be used in an expression between this type and another.
      * For example, an int and a double will return double.  Returns null if it doesn't know.
      */

    Type implicitCastFor (Type other)
    {
        throw new Error (fmt (this.classinfo.name ~ ".implicitCastFor (" ~ other.classinfo.name ~ "), %.*s to %.*s", toString (), other.toString ()));
        return null;
    }

    Type semantic0 (Scope scope)
    {
        return this;
    }

    Type semantic1 (Scope scope)
    {
        return this;
    }

    Type semantic2 (Scope scope)
    {
        return this;
    }

    Expression field (Marker mark, char [] name)
    {
        throw new Error (this.classinfo.name ~ ".field (\"" ~ name ~ "\") unimplemented.");
        return null;
    }
}

/** The root type of the basic types. */
class BasicType : Type
{
    override Type implicitCastFor (Type other)
    {
        if (isInteger ())
        {
            if (other.isReal ())
                return other;
            if (other.isInteger () && other.cellSize () < cellSize ())
                return this;
            if (other.isInteger ())
                return other;
            return super.implicitCastFor (other);
        }
        else if (isBaseReal ())
        {
            if (other.isInteger ())
                return this;
            else if (other.isBaseReal ())
                return (other.cellSize () < cellSize ()) ? this : other;
            else
                return super.implicitCastFor (other);
        }
        else
            return super.implicitCastFor (other);
    }
}

/** Boolean type. */
class BitType : BasicType
{
    static BitType singleton;

    static this () { singleton = new BitType (); }
    override int cellSize () { return 1; }
    override char [] toString () { return "bit"; }
}

/** Signed 8-bit integer. */
class ByteType : BasicType
{
    static ByteType singleton;

    static this () { singleton = new ByteType (); }
    override int cellSize () { return 1; }
    override char [] toString () { return "byte"; }
    override bit isInteger () { return true; }
}

/** Character type. */
class CharType : BasicType
{
    static CharType singleton;

    static this () { singleton = new CharType (); }
    override int cellSize () { return 1; }
    override char [] toString () { return "char"; }
}

/** Complex type using real. */
class ComplexType : BasicType
{
    static ComplexType singleton;

    static this () { singleton = new ComplexType (); }
    override int cellSize () { return 20; }
    override char [] toString () { return "complex"; }
    override bit isReal () { return true; }
}

/** Double-sized floating-point type. */
class DoubleType : BasicType
{
    static DoubleType singleton;

    static this () { singleton = new DoubleType (); }
    override int cellSize () { return 8; }
    override char [] toString () { return "double"; }
    override bit isReal () { return true; }
    override bit isBaseReal () { return true; }

    override Expression field (Marker mark, char [] name)
    {
        if (name == "init") return new FloatExp (mark, real.nan, singleton);
        return super.field (mark, name);
    }
}

/** Floating-point type using maximum precision. */
class RealType : BasicType
{
    static RealType singleton;

    static this () { singleton = new RealType (); }
    override int cellSize () { return 10; }
    override char [] toString () { return "real"; }
    override bit isReal () { return true; }
    override bit isBaseReal () { return true; }

    override Expression field (Marker mark, char [] name)
    {
        if (name == "init") return new FloatExp (mark, real.nan, singleton);
        return super.field (mark, name);
    }
}

/** Single-sized floating-point type. */
class FloatType : BasicType
{
    static FloatType singleton;

    static this () { singleton = new FloatType (); }
    override int cellSize () { return 4; }
    override char [] toString () { return "float"; }
    override bit isReal () { return true; }
    override bit isBaseReal () { return true; }

    override Expression field (Marker mark, char [] name)
    {
        if (name == "init") return new FloatExp (mark, real.nan, singleton);
        return super.field (mark, name);
    }
}

/** Imaginary component of a complex value using real precision. */
class ImaginaryType : BasicType
{
    static ImaginaryType singleton;

    static this () { singleton = new ImaginaryType (); }
    override int cellSize () { return 10; }
    override char [] toString () { return "imaginary"; }
    override bit isReal () { return true; }
}

/** Signed 32-bit integer. */
class IntType : BasicType
{
    static IntType singleton;

    static this () { singleton = new IntType (); }
    override int cellSize () { return 4; }
    override char [] toString () { return "int"; }
    override bit isInteger () { return true; }

    override Expression field (Marker mark, char [] name)
    {
        if (name == "init") return new IntegerExp (mark, 0, singleton);
        return super.field (mark, name);
    }
}

/** Signed 64-bit integer. */
class LongType : BasicType
{
    static LongType singleton;

    static this () { singleton = new LongType (); }
    override int cellSize () { return 8; }
    override char [] toString () { return "long"; }
    override bit isInteger () { return true; }
}

/** Unsigned 8-bit integer. */
class UByteType : BasicType
{
    static UByteType singleton;

    static this () { singleton = new UByteType (); }
    override int cellSize () { return 1; }
    override char [] toString () { return "ubyte"; }
    override bit isInteger () { return true; }
}

/** Unsigned 32-bit integer. */
class UIntType : BasicType
{
    static UIntType singleton;

    static this () { singleton = new UIntType (); }
    override int cellSize () { return 4; }
    override char [] toString () { return "uint"; }
    override bit isInteger () { return true; }
}

/** Unsigned 64-bit integer. */
class ULongType : BasicType
{
    static ULongType singleton;

    static this () { singleton = new ULongType (); }
    override int cellSize () { return 8; }
    override char [] toString () { return "ulong"; }
    override bit isInteger () { return true; }
}

/** Unsigned 16-bit integer. */
class UShortType : BasicType
{
    static UShortType singleton;

    static this () { singleton = new UShortType (); }
    override int cellSize () { return 2; }
    override char [] toString () { return "ushort"; }
    override bit isInteger () { return true; }
}

/** Non-value type. */
class VoidType : BasicType
{
    static VoidType singleton;

    static this () { singleton = new VoidType (); }
    override int cellSize () { return -1; }
    override char [] toString () { return "void"; }
}

/** Dynamic array type. */
class ArrayType : Type
{
    /** Assign the next pointer. */
    this (Type next)
    {
        this.next = next;
    }

    override int cellSize () { return 8; }
    override char [] toString () { return next.toString () ~ " []"; }

    override Expression field (Marker mark, char [] name)
    {
        if (name == "init") return new NullExp (mark, this);
        return super.field (mark, name);
    }
}

/** Associative array type. */
class AArrayType : Type
{
    Type key; /**< Type of key entries. */

    /** Assign the parameters. */
    this (Type next, Type key)
    {
        this.next = next;
        this.key = key;
    }

    override int cellSize () { return 8; }
}

/** Static-sized array type. */
class SArrayType : Type
{
    import net.BurtonRadons.parse.expression;
    
    Expression length; /**< Length of the array. */

    /** Assign the parameters. */
    this (Type next, Expression length)
    {
        this.next = next;
        this.length = length;
    }

    override int cellSize ()  { throw new Error ("Unimplemented SArray.cellSize"); return 0; }
}

/** Pointer type; (next *). */
class PointerType : Type
{
    /** Assign the parameter. */
    this (Type next)
    {
        this.next = next;
    }

    override int cellSize () { return 4; }
    override char [] toString () { return next.toString () ~ " *"; }
}

/** Struct type. */
class StructType : Type
{
    import net.BurtonRadons.parse.declaration;

    StructDeclaration decl; /**< Declaration for this type. */
    
    /** Assign the parameter. */
    this (StructDeclaration decl)
    {
        this.decl = decl;
    }

    override int cellSize ()
    {
        return decl.cellSize ();
    }

    override Expression field (Marker mark, char [] name)
    {
        if (name == "init")
        {
            StructExp exp = new StructExp (mark);

            exp.type = this;
            return exp;
        }

        return super.field (mark, name);
    }
}

/** Class instance type. */
class ClassType : Type
{
    import net.BurtonRadons.parse.declaration;

    ClassDeclaration decl; /**< Declaration for this type. */

    /** Assign the parameter. */
    this (ClassDeclaration decl)
    {
        this.decl = decl;
    }

    override int cellSize () { return 4; }

    char [] toString ()
    {
        return decl.name.toString ();
    }

    override Expression field (Marker mark, char [] name)
    {
        if (name == "init")
            return new NullExp (mark, this);
        return super.field (mark, name);
    }
}

/** The type of a function; next is null for special functions (such as constructors). */
class FunctionType : Type
{
    import net.BurtonRadons.parse.declaration;

    Declaration decl;
    Symbol [] args;
    bit varargs; /**< Set if this is a varargs function. */

    override int cellSize () { return -1; }

    /** Assign the parameters; next is the return type. */
    this (Type next, Symbol [] args)
    {
        this.next = next;
        this.args = args;
    }
}

/** An untyped name resolved during the semantic pass. */
class SymbolType : Type
{
    char [] [] name; /**< Symbolic name, dot-separated. */

    /** Assign the parameter. */
    this (char [] [] name)
    {
        this.name = name;
    }

    override int cellSize () { return -1; }

    char [] toBaseString ()
    {
        char [] result = name [0];

        for (int c = 1; c < name.length; c ++)
            result ~= "." ~ name [c];

        return result;
    }

    override char [] toString ()
    {
        return "symbol->" ~ toBaseString ();
    }

    Type semantic1 (Scope scope)
    {
        Symbol symbol = scope.find (name [0]);

        if (symbol === null)
            throw new Error (fmt ("TODO ERROR: Symbol %.*s not found.", toBaseString ()));
        for (int c = 1; c < name.length; c ++)
        {
            assert (symbol.decl !== null);
            symbol = symbol.decl.scope.find (name [c]);
            assert (symbol !== null);
        }

        assert (symbol.type !== null);
        return symbol.type;
    }
}
